/*
 * Decompiled with CFR 0.152.
 */
package org.apache.polaris.service.config;

import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.enterprise.context.ApplicationScoped;
import jakarta.enterprise.event.Observes;
import jakarta.enterprise.event.Startup;
import jakarta.enterprise.inject.Instance;
import jakarta.enterprise.inject.Produces;
import java.lang.annotation.Annotation;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.polaris.core.config.FeatureConfiguration;
import org.apache.polaris.core.config.ProductionReadinessCheck;
import org.apache.polaris.core.credentials.connection.ConnectionCredentialVendor;
import org.apache.polaris.core.persistence.MetaStoreManagerFactory;
import org.apache.polaris.service.auth.AuthenticationConfiguration;
import org.apache.polaris.service.auth.AuthenticationRealmConfiguration;
import org.apache.polaris.service.auth.AuthenticationType;
import org.apache.polaris.service.catalog.validation.IcebergPropertiesValidation;
import org.apache.polaris.service.config.FeaturesConfiguration;
import org.apache.polaris.service.config.ReadinessConfiguration;
import org.apache.polaris.service.context.DefaultRealmContextResolver;
import org.apache.polaris.service.context.RealmContextResolver;
import org.apache.polaris.service.context.TestRealmContextResolver;
import org.apache.polaris.service.credentials.connection.AuthType;
import org.apache.polaris.service.metrics.MetricsConfiguration;
import org.apache.polaris.service.persistence.InMemoryPolarisMetaStoreManagerFactory;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

@ApplicationScoped
public class ProductionReadinessChecks {
    private static final Logger LOGGER = LoggerFactory.getLogger(ProductionReadinessChecks.class);
    private static final String WARNING_SIGN_UTF_8 = "\u0000\u26a0\ufe0f";
    private static final String SEVERE_SIGN_UTF_8 = "\u0000\ud83d\uded1";
    private static final String WARNING_SIGN_PLAIN = "!!!";
    private static final String SEVERE_SIGN_PLAIN = "***STOP***";

    public void warnOnFailedChecks(@Observes Startup event, Instance<ProductionReadinessCheck> checks, ReadinessConfiguration config) {
        List errors = checks.stream().flatMap(check -> check.getErrors().stream()).toList();
        if (!errors.isEmpty()) {
            boolean utf8 = Charset.defaultCharset().equals(StandardCharsets.UTF_8);
            String warning = utf8 ? WARNING_SIGN_UTF_8 : WARNING_SIGN_PLAIN;
            String severe = utf8 ? SEVERE_SIGN_UTF_8 : SEVERE_SIGN_PLAIN;
            boolean hasSevere = errors.stream().anyMatch(ProductionReadinessCheck.Error::severe);
            LOGGER.makeLoggingEventBuilder(hasSevere ? Level.ERROR : Level.WARN).log("{} Production readiness checks failed! Check the warnings below.", (Object)(hasSevere ? severe : warning));
            errors.forEach(error -> LOGGER.makeLoggingEventBuilder(error.severe() ? Level.ERROR : Level.WARN).log("- {} {} Offending configuration option: '{}'.", new Object[]{error.severe() ? severe : warning, error.message(), error.offendingProperty()}));
            LOGGER.makeLoggingEventBuilder(hasSevere ? Level.ERROR : Level.WARN).log("Refer to https://polaris.apache.org/in-dev/unreleased/configuring-polaris-for-production for more information.");
            if (hasSevere) {
                if (!config.ignoreSevereIssues()) {
                    throw new IllegalStateException("Severe production readiness issues detected, startup aborted!");
                }
                LOGGER.warn("{} severe production readiness issues detected, but user explicitly requested startup by setting polaris.readiness.ignore-severe-issues=true and accepts the risk of denial-of-service, data-loss, corruption and others !", (Object)severe);
            }
        }
    }

    @Produces
    public ProductionReadinessCheck checkUserPrincipalMetricTag(MetricsConfiguration config) {
        if (config.userPrincipalTag().enableInApiMetrics()) {
            return ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])new ProductionReadinessCheck.Error[]{ProductionReadinessCheck.Error.of((String)"Metrics configuration includes user principal name and this could have security implications.", (String)"polaris.metrics.user-principal-tag.enable-in-api-metrics")});
        }
        return ProductionReadinessCheck.OK;
    }

    @Produces
    public ProductionReadinessCheck checkUserPrincipalAndRealmIdMetricTags(MetricsConfiguration config) {
        if (config.userPrincipalTag().enableInApiMetrics() && config.realmIdTag().enableInApiMetrics()) {
            return ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])new ProductionReadinessCheck.Error[]{ProductionReadinessCheck.Error.of((String)"Metrics configuration includes both user principal name and realm id in tags and this could have performance implications.", (String)"polaris.metrics.user-principal-tag.enable-in-api-metrics")});
        }
        return ProductionReadinessCheck.OK;
    }

    @Produces
    public ProductionReadinessCheck checkTokenBrokers(AuthenticationConfiguration configuration) {
        ArrayList errors = new ArrayList();
        configuration.realms().forEach((realm, config) -> {
            if (config.type() != AuthenticationType.EXTERNAL) {
                if (config.tokenBroker().type().equals("rsa-key-pair")) {
                    if (config.tokenBroker().rsaKeyPair().map(AuthenticationRealmConfiguration.TokenBrokerConfiguration.RSAKeyPairConfiguration::publicKeyFile).isEmpty()) {
                        errors.add(ProductionReadinessCheck.Error.of((String)"A public key file wasn't provided and will be generated.", (String)"polaris.authentication.%stoken-broker.rsa-key-pair.public-key-file".formatted(ProductionReadinessChecks.authRealmSegment(realm))));
                    }
                    if (config.tokenBroker().rsaKeyPair().map(AuthenticationRealmConfiguration.TokenBrokerConfiguration.RSAKeyPairConfiguration::privateKeyFile).isEmpty()) {
                        errors.add(ProductionReadinessCheck.Error.of((String)"A private key file wasn't provided and will be generated.", (String)"polaris.authentication.%stoken-broker.rsa-key-pair.private-key-file".formatted(ProductionReadinessChecks.authRealmSegment(realm))));
                    }
                }
                if (config.tokenBroker().type().equals("symmetric-key") && config.tokenBroker().symmetricKey().flatMap(AuthenticationRealmConfiguration.TokenBrokerConfiguration.SymmetricKeyConfiguration::secret).isPresent()) {
                    errors.add(ProductionReadinessCheck.Error.of((String)"A symmetric key secret was provided through configuration rather than through a secret file.", (String)"polaris.authentication.%stoken-broker.symmetric-key.secret".formatted(ProductionReadinessChecks.authRealmSegment(realm))));
                }
            }
        });
        return ProductionReadinessCheck.of(errors);
    }

    @Produces
    public ProductionReadinessCheck checkMetastore(MetaStoreManagerFactory factory) {
        if (factory instanceof InMemoryPolarisMetaStoreManagerFactory) {
            return ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])new ProductionReadinessCheck.Error[]{ProductionReadinessCheck.Error.of((String)"The current metastore is intended for tests only.", (String)"polaris.persistence.type")});
        }
        return ProductionReadinessCheck.OK;
    }

    @Produces
    public ProductionReadinessCheck checkRealmResolver(Config config, RealmContextResolver resolver) {
        if (resolver instanceof TestRealmContextResolver) {
            return ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])new ProductionReadinessCheck.Error[]{ProductionReadinessCheck.Error.of((String)"The current realm context resolver is intended for tests only.", (String)"polaris.realm-context.type")});
        }
        if (resolver instanceof DefaultRealmContextResolver) {
            boolean userProvided;
            ConfigValue configValue = config.getConfigValue("polaris.realm-context.require-header");
            boolean bl = userProvided = configValue.getSourceOrdinal() > 250;
            if ("false".equals(configValue.getValue()) && !userProvided) {
                return ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])new ProductionReadinessCheck.Error[]{ProductionReadinessCheck.Error.of((String)"The realm context resolver is configured to map requests without a realm header to the default realm.", (String)"polaris.realm-context.require-header")});
            }
        }
        return ProductionReadinessCheck.OK;
    }

    private static String authRealmSegment(String realm) {
        return realm.equals("<default>") ? "" : realm + ".";
    }

    @Produces
    public ProductionReadinessCheck checkInsecureStorageSettings(FeaturesConfiguration featureConfiguration) {
        FeatureConfiguration insecure = FeatureConfiguration.ALLOW_INSECURE_STORAGE_TYPES;
        ArrayList<ProductionReadinessCheck.Error> errors = new ArrayList<ProductionReadinessCheck.Error>();
        if (Boolean.parseBoolean(featureConfiguration.defaults().get(insecure.key()))) {
            errors.add(ProductionReadinessCheck.Error.ofSevere((String)("Must not enable a configuration that exposes known and severe security risks: " + insecure.description()), (String)String.format("polaris.features.\"%s\"", insecure.key())));
        }
        featureConfiguration.realmOverrides().forEach((realmId, overrides) -> {
            if (Boolean.parseBoolean(overrides.overrides().get(insecure.key()))) {
                errors.add(ProductionReadinessCheck.Error.ofSevere((String)("Must not enable a configuration that exposes known and severe security risks: " + insecure.description()), (String)String.format("polaris.features.realm-overrides.\"%s\".overrides.\"%s\"", realmId, insecure.key())));
            }
        });
        FeatureConfiguration storageTypes = FeatureConfiguration.SUPPORTED_CATALOG_STORAGE_TYPES;
        ObjectMapper mapper = new ObjectMapper();
        Map<String, Object> defaults = featureConfiguration.parseDefaults(mapper);
        Map<String, Map<String, Object>> realmOverrides = featureConfiguration.parseRealmOverrides(mapper);
        List supported = defaults.getOrDefault(storageTypes.key(), List.of());
        supported.stream().filter(n -> !IcebergPropertiesValidation.safeStorageType(n)).forEach(t -> errors.add(ProductionReadinessCheck.Error.ofSevere((String)String.format("The storage type '%s' is considered insecure and exposes the service to severe security risks!", t), (String)String.format("polaris.features.\"%s\"", storageTypes.key()))));
        realmOverrides.forEach((realmId, overrides) -> {
            List s = overrides.getOrDefault(storageTypes.key(), List.of());
            s.stream().filter(n -> !IcebergPropertiesValidation.safeStorageType(n)).forEach(t -> errors.add(ProductionReadinessCheck.Error.ofSevere((String)String.format("The storage type '%s' is considered insecure and exposes the service to severe security risks!", t), (String)String.format("polaris.features.realm-overrides.\"%s\".overrides.\"%s\"", realmId, storageTypes.key()))));
        });
        return errors.isEmpty() ? ProductionReadinessCheck.OK : ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])errors.toArray(new ProductionReadinessCheck.Error[0]));
    }

    @Produces
    public ProductionReadinessCheck checkOverlappingSiblingCheckSettings(FeaturesConfiguration featureConfiguration) {
        FeatureConfiguration optimizedSiblingCheck = FeatureConfiguration.OPTIMIZED_SIBLING_CHECK;
        FeatureConfiguration allowOverlap = FeatureConfiguration.ALLOW_OPTIMIZED_SIBLING_CHECK;
        ArrayList<ProductionReadinessCheck.Error> errors = new ArrayList<ProductionReadinessCheck.Error>();
        if (Boolean.parseBoolean(featureConfiguration.defaults().get(optimizedSiblingCheck.key())) && !Boolean.parseBoolean(featureConfiguration.defaults().get(allowOverlap.key()))) {
            errors.add(ProductionReadinessCheck.Error.ofSevere((String)"This setting should be used with care and only enabled in new realms. Enabling it in previously used realms and may lead to incorrect behavior, due to missing data for location_without_scheme column. Set the ALLOW_OPTIMIZED_SIBLING_CHECK flag to acknowledge this warning and enable Polaris to start.", (String)String.format("polaris.features.\"%s\"", optimizedSiblingCheck.key())));
        }
        featureConfiguration.realmOverrides().forEach((realmId, overrides) -> {
            if (Boolean.parseBoolean(overrides.overrides().get(optimizedSiblingCheck.key())) && !Boolean.parseBoolean(overrides.overrides().get(allowOverlap.key()))) {
                errors.add(ProductionReadinessCheck.Error.ofSevere((String)"This setting should be used with care and only enabled in new realms. Enabling it in previously used realms and may lead to incorrect behavior, due to missing data for location_without_scheme column. Set the ALLOW_OPTIMIZED_SIBLING_CHECK flag to acknowledge this warning and enable Polaris to start.", (String)String.format("polaris.features.realm-overrides.\"%s\".overrides.\"%s\"", realmId, optimizedSiblingCheck.key())));
            }
        });
        return errors.isEmpty() ? ProductionReadinessCheck.OK : ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])errors.toArray(new ProductionReadinessCheck.Error[0]));
    }

    @Produces
    public ProductionReadinessCheck checkConnectionCredentialVendors(Instance<ConnectionCredentialVendor> credentialVendors, FeaturesConfiguration featureConfiguration) {
        ObjectMapper mapper = new ObjectMapper();
        Map<String, Object> defaults = featureConfiguration.parseDefaults(mapper);
        Map<String, Map<String, Object>> realmOverrides = featureConfiguration.parseRealmOverrides(mapper);
        String federationKey = FeatureConfiguration.ENABLE_CATALOG_FEDERATION.key();
        String authTypesKey = FeatureConfiguration.SUPPORTED_EXTERNAL_CATALOG_AUTHENTICATION_TYPES.key();
        boolean federationEnabled = Boolean.parseBoolean(defaults.getOrDefault(federationKey, false).toString());
        List defaultAuthTypes = defaults.getOrDefault(authTypesKey, List.of());
        HashSet allAuthTypes = new HashSet();
        if (federationEnabled) {
            allAuthTypes.addAll(defaultAuthTypes);
        }
        realmOverrides.forEach((id, overrides) -> {
            if (Boolean.parseBoolean(((Object)overrides.getOrDefault(federationKey, federationEnabled)).toString())) {
                allAuthTypes.addAll(overrides.containsKey(authTypesKey) ? overrides.get(authTypesKey) : defaultAuthTypes);
            }
        });
        ArrayList<ProductionReadinessCheck.Error> errors = new ArrayList<ProductionReadinessCheck.Error>();
        for (String name : allAuthTypes) {
            try {
                org.apache.polaris.core.connection.AuthenticationType type = org.apache.polaris.core.connection.AuthenticationType.valueOf((String)name);
                if (!credentialVendors.select(new Annotation[]{AuthType.Literal.of(type)}).isUnsatisfied()) continue;
                errors.add(ProductionReadinessCheck.Error.ofSevere((String)String.format("Catalog federation is enabled but no ConnectionCredentialVendor found for authentication type '%s'. External catalog connections using this authentication type will fail.", type), (String)String.format("polaris.features.\"%s\"", authTypesKey)));
            }
            catch (IllegalArgumentException e) {
                errors.add(ProductionReadinessCheck.Error.ofSevere((String)String.format("Invalid authentication type '%s' in SUPPORTED_EXTERNAL_CATALOG_AUTHENTICATION_TYPES configuration.", name), (String)String.format("polaris.features.\"%s\"", authTypesKey)));
            }
        }
        return errors.isEmpty() ? ProductionReadinessCheck.OK : ProductionReadinessCheck.of((ProductionReadinessCheck.Error[])errors.toArray(new ProductionReadinessCheck.Error[0]));
    }
}

