/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.ChunkExtensionProvider;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.TrailerProvider;
import software.amazon.awssdk.utils.Pair;
import software.amazon.awssdk.utils.Validate;
import software.amazon.awssdk.utils.async.AddingTrailingDataSubscriber;
import software.amazon.awssdk.utils.async.ContentLengthAwareSubscriber;
import software.amazon.awssdk.utils.async.DelegatingSubscriber;
import software.amazon.awssdk.utils.async.FlatteningSubscriber;
import software.amazon.awssdk.utils.internal.MappingSubscriber;

@SdkInternalApi
public class ChunkedEncodedPublisher
implements Publisher<ByteBuffer> {
    private static final byte[] CRLF = new byte[]{13, 10};
    private static final byte SEMICOLON = 59;
    private static final byte EQUALS = 61;
    private static final byte COLON = 58;
    private static final byte COMMA = 44;
    private final Publisher<ByteBuffer> wrapped;
    private final long contentLength;
    private final List<ChunkExtensionProvider> extensions = new ArrayList<ChunkExtensionProvider>();
    private final List<TrailerProvider> trailers = new ArrayList<TrailerProvider>();
    private final int chunkSize;
    private ByteBuffer chunkBuffer;
    private final boolean addEmptyTrailingChunk;

    public ChunkedEncodedPublisher(Builder b) {
        this.wrapped = b.publisher;
        this.contentLength = (Long)Validate.notNull((Object)b.contentLength, (String)"contentLength must not be null", (Object[])new Object[0]);
        this.chunkSize = b.chunkSize;
        this.extensions.addAll(b.extensions);
        this.trailers.addAll(b.trailers);
        this.addEmptyTrailingChunk = b.addEmptyTrailingChunk;
    }

    public void subscribe(Subscriber<? super ByteBuffer> subscriber) {
        Publisher<ByteBuffer> lengthEnforced = this.limitLength(this.wrapped, this.contentLength);
        Publisher<Iterable<ByteBuffer>> chunked = this.chunk(lengthEnforced);
        Publisher<Iterable<ByteBuffer>> trailingAdded = this.addTrailingChunks(chunked);
        Publisher<ByteBuffer> flattened = this.flatten(trailingAdded);
        Publisher<ByteBuffer> encoded = this.map(flattened, this::encodeChunk);
        encoded.subscribe(subscriber);
    }

    public static Builder builder() {
        return new Builder();
    }

    private Iterable<Iterable<ByteBuffer>> getTrailingChunks() {
        ArrayList<ByteBuffer> trailing = new ArrayList<ByteBuffer>();
        if (this.chunkBuffer != null) {
            this.chunkBuffer.flip();
            if (this.chunkBuffer.hasRemaining()) {
                trailing.add(this.chunkBuffer);
            }
        }
        if (this.addEmptyTrailingChunk) {
            trailing.add(ByteBuffer.allocate(0));
        }
        return Collections.singletonList(trailing);
    }

    private Publisher<ByteBuffer> limitLength(Publisher<ByteBuffer> publisher, long length) {
        return subscriber -> publisher.subscribe((Subscriber)new ContentLengthAwareSubscriber(subscriber, length));
    }

    private Publisher<Iterable<ByteBuffer>> chunk(Publisher<ByteBuffer> upstream) {
        return subscriber -> upstream.subscribe((Subscriber)new ChunkingSubscriber((Subscriber<? super Iterable<ByteBuffer>>)subscriber));
    }

    private Publisher<ByteBuffer> flatten(Publisher<Iterable<ByteBuffer>> upstream) {
        return subscriber -> upstream.subscribe((Subscriber)new FlatteningSubscriber(subscriber));
    }

    public Publisher<Iterable<ByteBuffer>> addTrailingChunks(Publisher<Iterable<ByteBuffer>> upstream) {
        return subscriber -> upstream.subscribe((Subscriber)new AddingTrailingDataSubscriber(subscriber, this::getTrailingChunks));
    }

    public Publisher<ByteBuffer> map(Publisher<ByteBuffer> upstream, Function<? super ByteBuffer, ? extends ByteBuffer> mapper) {
        return subscriber -> upstream.subscribe((Subscriber)MappingSubscriber.create((Subscriber)subscriber, (Function)mapper));
    }

    private ByteBuffer encodeChunk(ByteBuffer byteBuffer) {
        int contentLen = byteBuffer.remaining();
        byte[] chunkSizeHex = Integer.toHexString(byteBuffer.remaining()).getBytes(StandardCharsets.UTF_8);
        List<Pair<byte[], byte[]>> chunkExtensions = this.extensions.stream().map(e -> {
            ByteBuffer duplicate = byteBuffer.duplicate();
            return e.get(duplicate);
        }).collect(Collectors.toList());
        int extensionsLength = this.calculateExtensionsLength(chunkExtensions);
        boolean isTrailerChunk = contentLen == 0;
        List<Object> trailerData = isTrailerChunk ? this.getTrailerData() : Collections.emptyList();
        int trailerLen = trailerData.stream().mapToInt(t -> t.remaining() + CRLF.length).sum();
        int encodedLen = chunkSizeHex.length + extensionsLength + CRLF.length + contentLen + trailerLen + CRLF.length;
        if (isTrailerChunk) {
            encodedLen += CRLF.length;
        }
        ByteBuffer encoded = ByteBuffer.allocate(encodedLen);
        encoded.put(chunkSizeHex);
        chunkExtensions.forEach(p -> {
            encoded.put((byte)59);
            encoded.put((byte[])p.left());
            if (p.right() != null && ((byte[])p.right()).length > 0) {
                encoded.put((byte)61);
                encoded.put((byte[])p.right());
            }
        });
        encoded.put(CRLF);
        if (byteBuffer.hasRemaining()) {
            encoded.put(byteBuffer);
            encoded.put(CRLF);
        }
        if (isTrailerChunk) {
            trailerData.forEach(t -> {
                encoded.put((ByteBuffer)t);
                encoded.put(CRLF);
            });
            encoded.put(CRLF);
        }
        encoded.flip();
        return encoded;
    }

    private int calculateExtensionsLength(List<Pair<byte[], byte[]>> chunkExtensions) {
        return chunkExtensions.stream().mapToInt(p -> {
            int keyLen = ((byte[])p.left()).length;
            byte[] value = (byte[])p.right();
            if (value.length > 0) {
                return 1 + keyLen + 1 + value.length;
            }
            return 1 + keyLen;
        }).sum();
    }

    private List<ByteBuffer> getTrailerData() {
        ArrayList<ByteBuffer> data = new ArrayList<ByteBuffer>();
        for (TrailerProvider provider : this.trailers) {
            Pair<String, List<String>> trailer = provider.get();
            byte[] key = ((String)trailer.left()).getBytes(StandardCharsets.UTF_8);
            List values = ((List)trailer.right()).stream().map(v -> v.getBytes(StandardCharsets.UTF_8)).collect(Collectors.toList());
            if (values.isEmpty()) {
                throw new RuntimeException(String.format("Trailing header '%s' has no values", trailer.left()));
            }
            int valuesLen = values.stream().mapToInt(v -> ((byte[])v).length).sum();
            int size = key.length + 1 + valuesLen + values.size() - 1;
            ByteBuffer trailerData = ByteBuffer.allocate(size);
            trailerData.put(key);
            trailerData.put((byte)58);
            for (int i = 0; i < values.size(); ++i) {
                trailerData.put((byte[])values.get(i));
                if (i + 1 == values.size()) continue;
                trailerData.put((byte)44);
            }
            trailerData.flip();
            data.add(trailerData);
        }
        return data;
    }

    public static class Builder {
        private Publisher<ByteBuffer> publisher;
        private Long contentLength;
        private int chunkSize;
        private boolean addEmptyTrailingChunk;
        private final List<ChunkExtensionProvider> extensions = new ArrayList<ChunkExtensionProvider>();
        private final List<TrailerProvider> trailers = new ArrayList<TrailerProvider>();

        public Builder publisher(Publisher<ByteBuffer> publisher) {
            this.publisher = publisher;
            return this;
        }

        public Publisher<ByteBuffer> publisher() {
            return this.publisher;
        }

        public Builder contentLength(long contentLength) {
            this.contentLength = contentLength;
            return this;
        }

        public Builder chunkSize(int chunkSize) {
            this.chunkSize = chunkSize;
            return this;
        }

        public Builder addEmptyTrailingChunk(boolean addEmptyTrailingChunk) {
            this.addEmptyTrailingChunk = addEmptyTrailingChunk;
            return this;
        }

        public Builder addExtension(ChunkExtensionProvider extension) {
            this.extensions.add(extension);
            return this;
        }

        public Builder addTrailer(TrailerProvider trailerProvider) {
            this.trailers.add(trailerProvider);
            return this;
        }

        public List<TrailerProvider> trailers() {
            return this.trailers;
        }

        public ChunkedEncodedPublisher build() {
            return new ChunkedEncodedPublisher(this);
        }
    }

    private class ChunkingSubscriber
    extends DelegatingSubscriber<ByteBuffer, Iterable<ByteBuffer>> {
        protected ChunkingSubscriber(Subscriber<? super Iterable<ByteBuffer>> subscriber) {
            super(subscriber);
        }

        public void onNext(ByteBuffer byteBuffer) {
            if (ChunkedEncodedPublisher.this.chunkBuffer == null) {
                ChunkedEncodedPublisher.this.chunkBuffer = ByteBuffer.allocate(ChunkedEncodedPublisher.this.chunkSize);
            }
            long totalBufferedBytes = (long)ChunkedEncodedPublisher.this.chunkBuffer.position() + (long)byteBuffer.remaining();
            int nBufferedChunks = (int)(totalBufferedBytes / (long)ChunkedEncodedPublisher.this.chunkSize);
            ArrayList<ByteBuffer> chunks = new ArrayList<ByteBuffer>(nBufferedChunks);
            if (nBufferedChunks > 0) {
                for (int i = 0; i < nBufferedChunks; ++i) {
                    ByteBuffer slice = byteBuffer.slice();
                    int maxBytesToCopy = Math.min(ChunkedEncodedPublisher.this.chunkBuffer.remaining(), slice.remaining());
                    slice.limit(maxBytesToCopy);
                    ChunkedEncodedPublisher.this.chunkBuffer.put(slice);
                    if (!ChunkedEncodedPublisher.this.chunkBuffer.hasRemaining()) {
                        ChunkedEncodedPublisher.this.chunkBuffer.flip();
                        chunks.add(ChunkedEncodedPublisher.this.chunkBuffer);
                        ChunkedEncodedPublisher.this.chunkBuffer = ByteBuffer.allocate(ChunkedEncodedPublisher.this.chunkSize);
                    }
                    byteBuffer.position(byteBuffer.position() + maxBytesToCopy);
                }
                if (byteBuffer.hasRemaining()) {
                    ChunkedEncodedPublisher.this.chunkBuffer.put(byteBuffer);
                }
            } else {
                ChunkedEncodedPublisher.this.chunkBuffer.put(byteBuffer);
            }
            this.subscriber.onNext(chunks);
        }
    }
}

